<?php

namespace Drupal\insert\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Form\ConfigFormBase;

/**
 *
 */
class ConfigForm extends ConfigFormBase {

  /**
   * @inheritdoc
   */
  public function __construct(ConfigFactoryInterface $config_factory) {
    parent::__construct($config_factory);
  }

  /**
   * @inheritdoc
   */
  public function getFormId() {
    return 'insert_config_form';
  }

  /**
   * @inheritdoc
   */
  protected function getEditableConfigNames() {
    return ['insert.config'];
  }

  /**
   * @inheritdoc
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('insert.config');

    $text_formats = array_map(function ($format) {
      /** @var \Drupal\filter\Entity\FilterFormat $format */
      return $format->label();
    }, filter_formats());

    $form['text_formats'] = [
      '#type' => 'checkboxes',
      '#options' => $text_formats,
      '#title' => $this->t('Automatic text format support'),
      '#description' => $this->t('Drupal core\'s HTML filter removes tags and attributes not explicitly white-listed from the output. This might strip tags and attributes generated by the Insert module. Enabling automatic text format support alters allowed HTML tags of the HTML filter when saving a text format configuration. Since managing the tags and attributes is prone to errors, enabling text format support here will ensure that required tags and attributes are always set. <strong>Note: After enabling text format support, you need to save the corresponding text format(s) at least once for the necessary tags to be added. When disabling support of a text format tags and attributes added to the format are not removed, they just do not get added automatically anymore when saving the text format(s).</strong>'),
      '#default_value' => $config->get('text_formats')
      ? $config->get('text_formats') : [],
    ];

    $form['absolute'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Use absolute paths'),
      '#description' => $this->t('Includes the full URL prefix "@base_url" in all links and image tags.', ['@base_url' => $GLOBALS['base_url']]),
      '#default_value' => $config->get('absolute') ? '1' : '0',
    ];

    $form['file_field_images_enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Allow inserting &lt;img&gt; tags for images uploaded to generic file fields'),
      '#description' => $this->t('By activating this option, image styles may be selected in the Insert settings on (generic) file field widgets. Then, such may be used to insert images uploaded to file fields, while other files still may be inserted using compatible Insert styles only.'),
      '#default_value' => $config->get('file_field_images_enabled') ? '1' : '0',
    ];

    $form['widgets'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Supported widgets'),
      '#description' => $this->t('Lists of field widget ids that Insert should be available to. Separate widget ids with a space or comma.'),
    ];

    $form['widgets']['widgets__' . INSERT_TYPE_FILE] = [
      '#type' => 'textfield',
      '#title' => $this->t('File widgets'),
      '#default_value' => join(', ', $config->get('widgets.file')),
      '#element_validate' => [[get_called_class(), 'string_to_array']],
    ];

    $form['widgets']['widgets__' . INSERT_TYPE_IMAGE] = [
      '#type' => 'textfield',
      '#title' => $this->t('Image widgets'),
      '#default_value' => join(', ', $config->get('widgets.image')),
      '#element_validate' => [[get_called_class(), 'string_to_array']],
    ];

    $form['css_classes'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Additional CSS classes'),
      '#description' => $this->t('CSS classes to be added to items inserted using the Insert module. Separate classes with a space or comma.'),
    ];

    $form['css_classes'][INSERT_TYPE_FILE] = [
      '#type' => 'textfield',
      '#title' => $this->t('Classes to be added to generic file links'),
      '#default_value' => join(' ', $config->get('css_classes.file')),
      '#element_validate' => [[get_called_class(), 'string_to_array']],
    ];

    $form['css_classes'][INSERT_TYPE_IMAGE] = [
      '#type' => 'textfield',
      '#title' => $this->t('Classes to be added to images and image links'),
      '#default_value' => join(' ', $config->get('css_classes.image')),
      '#element_validate' => [[get_called_class(), 'string_to_array']],
    ];

    $form['file_extensions'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('File type detection'),
      '#description' => $this->t('In order to have Insert detect particular file types (i.e. to allow embedding media files), the module needs be aware of which file extensions map to such particular file types. Be sure to enable these file extensions for upload in the file field settings as well. Separate extensions with a space or comma and do not include the leading dot.'),
    ];

    $form['file_extensions']['audio'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Audio file extensions'),
      '#default_value' => join(', ', $config->get('file_extensions.audio')),
      '#element_validate' => [[get_called_class(), 'string_to_array']],
    ];

    $form['file_extensions']['video'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Video file extensions'),
      '#default_value' => join(', ', $config->get('file_extensions.video')),
      '#element_validate' => [[get_called_class(), 'string_to_array']],
    ];

    $form += \Drupal::moduleHandler()->invokeAll(
      'insert_config_form',
      [$form]
    );

    return parent::buildForm($form, $form_state);
  }

  /**
   * @inheritdoc
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $config = $this->config('insert.config');

    $config->set('text_formats', array_keys(
      array_filter($form_state->getValue('text_formats'), function ($value) {
        return !!$value;
      })
    ));
    $config->set('absolute', !!$form_state->getValue('absolute'));
    $config->set('file_field_images_enabled', !!$form_state->getValue('file_field_images_enabled'));

    $config->set('widgets.' . INSERT_TYPE_FILE, $this->sanitize_array($form_state->getValue('widgets__' . INSERT_TYPE_FILE)));
    $config->set('widgets.' . INSERT_TYPE_IMAGE, $this->sanitize_array($form_state->getValue('widgets__' . INSERT_TYPE_IMAGE)));

    $config->set('css_classes.' . INSERT_TYPE_FILE, $this->sanitize_array($form_state->getValue(INSERT_TYPE_FILE)));
    $config->set('css_classes.' . INSERT_TYPE_IMAGE, $this->sanitize_array($form_state->getValue(INSERT_TYPE_IMAGE)));

    $config->set('file_extensions.audio', $form_state->getValue('audio'));
    $config->set('file_extensions.video', $form_state->getValue('video'));

    $config->save();

    \Drupal::moduleHandler()->invokeAll(
      'insert_config_submit_form',
      [$form_state]
    );

    parent::submitForm($form, $form_state);
  }

  /**
   * @param array $source
   * @return array
   */
  protected function sanitize_array(array $source) {
    return array_filter($source, function ($value) {
      return $value !== '';
    });
  }

  /**
   * @param array $element
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   */
  public static function string_to_array(array $element, &$form_state) {
    $value = str_replace(',', ' ', trim($element['#value']));
    $form_state->setValueForElement($element, preg_split('/\s+/', $value));
  }

}
